'use client'
import { useState, useEffect } from 'react';
import Layout from "@/components/Layout/LayoutAdmin";
import StatusCard from "@/components/StatusCard";
import SearchActivityChart from "@/components/SearchActivityChart";
import axios from "axios";
import Image from "next/image";
import { search, EyeIcon, pause, edit, totalsearch, violations, falsepositiverate } from '@/assets/images'
import { useAuth } from "@/context/AuthContext";
import { formatDateWithOrdinal } from '@/utils/dateFormat';

export default function CoumponManagement() {
  const [results, setResults] = useState([]);
  const apiUrl = process.env.NEXT_PUBLIC_API_URL;
  const { user, loading: authLoading } = useAuth();

  // Pagination states
  const [currentPage, setCurrentPage] = useState(1);
  const [itemsPerPage] = useState(10); // Number of items to display per page

  const quickCards = [
    {
      title: 'Total Searches ',
      result: '24,583',
      icon: totalsearch
    },
    {
      title: 'Violations Detected',
      result: '783',
      icon: violations
    },
    {
      title: 'False positive rate',
      result: '3.2%',
      icon: falsepositiverate
    },
  ];

  useEffect(() => {
    const fetchActiveUsers = async () => {
      if (authLoading || !user?.token) {
        // Do not fetch if auth is still loading or user token is not available
        return;
      }

      try {
        const response = await axios.get(`${apiUrl}/getdata/top-active-users`, {
          headers: {
            Authorization: `Bearer ${user.token}`,
          },
        });
        setResults(response.data || []); // Assuming response.data is an array of users
        setCurrentPage(1); // Reset to first page on new data fetch
      } catch (err) {
        setResults([]);
        console.error("Error fetching active users:", err);
      }
    };
    fetchActiveUsers(); // Call the fetch function
  }, [apiUrl, user, authLoading]); // Re-run effect if apiUrl, user, or authLoading changes

  // Calculate current users for the table
  const indexOfLastItem = currentPage * itemsPerPage;
  const indexOfFirstItem = indexOfLastItem - itemsPerPage;
  const currentUsers = results.slice(indexOfFirstItem, indexOfLastItem);

  // Calculate total pages
  const totalPages = Math.ceil(results.length / itemsPerPage);

  // Handle page change
  const handlePageChange = (pageNumber) => {
    setCurrentPage(pageNumber);
  };

  return (
    <Layout>
      <div className="max-w-full mx-auto my-10">
        <ul className="flex flex-wrap -mx-2.5 mt-5 mb-10">
          {quickCards.map((card, c) => (
            <li
              key={c}
              className="w-full xs:w-1/2 md:w-1/3 p-2.5"
            >
              <StatusCard data={card} />
            </li>
          ))}
        </ul>

        <SearchActivityChart />

        <h3 className="mb-5 font-semibold text-[#373737] mt-5 pt-3 text-2xl">Top Active users</h3>
        <div className="table_wpr pb-5">
          <div className="relative overflow-x-auto">
            <table className="glob_table ">
              <thead>
                <tr>
                  <th>Sl. No.</th>
                  <th>User Name</th>
                  <th className='text-center'>Number of Rescans</th>
                  <th>Last Login</th>
                </tr>
              </thead>
              <tbody>
                {currentUsers.length > 0 ? (
                  currentUsers.map((result, r) => (
                    <tr key={result.id || r}> {/* Use a unique ID if available, fallback to index */}
                      <td className="max-w-32">{indexOfFirstItem + r + 1}</td>
                      <td className="max-w-32">{result.name}</td>
                      <td className='text-center'>{result.searchResultCount}</td>
                      <td>{formatDateWithOrdinal(result.lastActivity)}</td>
                    </tr>
                  ))
                ) : (
                  <tr>
                    <td colSpan="4" className="text-center py-4">
                      {authLoading ? "Loading users..." : "No active users found."}
                    </td>
                  </tr>
                )}
              </tbody>
            </table>
          </div>
        </div>

        {totalPages > 1 && ( // Only show pagination if there's more than one page
          <div className="flex justify-end mt-8">
            <ul className="flex items-center space-x-2">
              <li
                onClick={() => handlePageChange(currentPage - 1)}
                className={`px-5 h-8 border border-[#c9c9c9] flex items-center justify-center bg-white text-black text-sm font-medium rounded cursor-pointer ${
                  currentPage === 1 ? 'opacity-50 cursor-not-allowed' : ''
                }`}
                style={{ pointerEvents: currentPage === 1 ? 'none' : 'auto' }} // Disable pointer events when not allowed
              >
                Previous
              </li>
              {Array.from({ length: totalPages }, (_, i) => i + 1).map((page) => (
                <li
                  key={page}
                  onClick={() => handlePageChange(page)}
                  className={`w-8 h-8 border border-[#c9c9c9] flex items-center justify-center text-sm font-medium rounded cursor-pointer ${
                    currentPage === page ? 'bg-primary text-white' : 'bg-white text-black'
                  }`}
                >
                  {page}
                </li>
              ))}
              <li
                onClick={() => handlePageChange(currentPage + 1)}
                className={`px-5 h-8 border border-[#c9c9c9] flex items-center justify-center bg-white text-black text-sm font-medium rounded cursor-pointer ${
                  currentPage === totalPages ? 'opacity-50 cursor-not-allowed' : ''
                }`}
                style={{ pointerEvents: currentPage === totalPages ? 'none' : 'auto' }} // Disable pointer events when not allowed
              >
                Next
              </li>
            </ul>
          </div>
        )}
      </div>
    </Layout>
  );
}