'use client'
import Layout from "@/components/Layout/Layout";
import StatusScale from "@/components/StatusScale";
import Image from "next/image";
import { useEffect, useRef, useState } from "react";
import { reportIcon } from '@/assets/images'
import Modal from "@/components/Modal";
import axios from "axios";
import { useSearchParams } from 'next/navigation';
import { useAuth } from "@/context/AuthContext";

export default function Search() {
  const searchParams = useSearchParams();
  const bookid = searchParams.get('bookid');

  const [isOpen, setIsOpen] = useState(false);
  const modalRef = useRef(null);
  const [book, setBook] = useState(null);
  const [results, setResults] = useState([]);
  const [loadingA, setLoadingA] = useState(true);
  const [currentPage, setCurrentPage] = useState(1);
  const resultsPerPage = 10;
  const { user, loading } = useAuth();
  const authID = user ? user.userId : 0;

  const apiUrl = process.env.NEXT_PUBLIC_API_URL;

  const fetchBook = async () => {
    try {
      const response = await axios.get(`${apiUrl}/getdata?tab=uploads&authID=${authID}&keyName=_id&keyValue=${bookid}`, {
          headers: {
            Authorization: `Bearer ${user?.token}`
          }
        });
      setBook(response.data[0] || null);
    } catch (err) {
      setBook(null);
    }
  };

  const fetchBookSearch = async () => {
    try {
      setLoadingA(true);
      const response = await axios.get(`${apiUrl}/getdata/searchresult?itemid=${bookid}`, {
          headers: {
            Authorization: `Bearer ${user?.token}`
          }
        });
      setResults(response.data || []);
    } catch (err) {
      setResults([]);
    }
    setLoadingA(false);
  };

  useEffect(() => {
    function handleClickOutside(event) {
      if (modalRef.current && !modalRef.current.contains(event.target)) {
        setIsOpen(false);
      }
    }
    if (!loading && user) {
      fetchBook();
      fetchBookSearch();
    }
    if (isOpen) {
      document.addEventListener("mousedown", handleClickOutside);
      document.body.classList.add("overflow-hidden");
    } else {
      document.body.classList.remove("overflow-hidden");
    }
    return () => {
      document.removeEventListener("mousedown", handleClickOutside);
    };
    // eslint-disable-next-line
  }, [isOpen, bookid, authID, loading, user]);

  // Pagination logic
  const indexOfLastResult = currentPage * resultsPerPage;
  const indexOfFirstResult = indexOfLastResult - resultsPerPage;
  const currentResults = results.slice(indexOfFirstResult, indexOfLastResult);
  const totalPages = Math.ceil(results.length / resultsPerPage);

  const handlePageChange = (page) => {
    if (page >= 1 && page <= totalPages) setCurrentPage(page);
  };

  return (
    <Layout>
      <div className="max-w-[1064px] mx-auto my-10">
        <div className="flex flex-wrap items-end gap-3 mb-5">
          <div className="mr-auto">
            <h3 className="mb-1.5 font-semibold text-[#373737] text-2xl">Book Name</h3>
            {loadingA ? (
              <p className="text-sm text-gray-500">Loading...</p>
            ) : (
              <p className="text-sm leading-5 font-normal text-[#575757]">
                {book ? book.name || book.title : "Not found"}
              </p>
            )}
          </div>
          <div className="flex items-center">
            <h5 className="text-xs lg:text-sm font-normal text-[#575757] pr-2 mr-2 lg:pr-3 lg:mr-3 border-r border-gray-300">Matches Found</h5>
            <h2 className="text-xl lg:text-2xl leading-none font-bold text-[#404040]">{results.length}</h2>
          </div>
        </div>
        <div className="table_wpr">
          <div className="relative overflow-x-auto">
            <table className="glob_table">
              <thead>
                <tr>
                  <th>Scan Key</th>
                  <th className="text-center">YouTube Video</th>
                  <th className="text-center">Link</th>
                  <th className="text-center">Channel Name</th>
                  <th className="text-center">Report False Positive</th>
                </tr>
              </thead>
              <tbody>
                {loadingA ? (
                  <tr>
                    <td colSpan={7} className="text-center">Loading...</td>
                  </tr>
                ) : currentResults.length === 0 ? (
                  <tr>
                    <td colSpan={7} className="text-center">No results found.</td>
                  </tr>
                ) : (
                  currentResults.map((result, r) => (
                    <tr key={r}>
                      <td>{result.keyword}</td>
                      <td className="text-center">{result.video}</td>
                      <td className="text-center"><a href={result.link} className="text-primary" target="_blank" rel="noopener noreferrer">Watch now</a></td>
                      <td className="text-center"><a href={result.channel?.link} className="text-primary" target="_blank" rel="noopener noreferrer">{result.channel?.name}</a></td>
                      <td>
                        <div className="flex justify-center">
                          <button type="button" onClick={() => setIsOpen(!isOpen)}>
                            <Image src={reportIcon} height={24} width={24} alt="Report false" className="p-1 cursor-pointer" />
                          </button>
                        </div>
                      </td>
                    </tr>
                  ))
                )}
              </tbody>
            </table>
          </div>
        </div>
        {/* Pagination Controls */}
        <div className="flex justify-end mt-8">
          <ul className="flex items-center space-x-2">
            <li
              className={`px-3 h-8 border border-[#c9c9c9] flex items-center justify-center bg-white text-black text-sm font-medium rounded cursor-pointer ${currentPage === 1 ? 'opacity-50 cursor-not-allowed' : ''}`}
              onClick={() => handlePageChange(currentPage - 1)}
            >
              Prev
            </li>
            {[...Array(totalPages)].map((_, idx) => (
              <li
                key={idx}
                className={`w-8 h-8 border border-[#c9c9c9] flex items-center justify-center ${currentPage === idx + 1 ? 'bg-primary text-white' : 'bg-white text-black'} text-sm font-medium rounded cursor-pointer`}
                onClick={() => handlePageChange(idx + 1)}
              >
                {idx + 1}
              </li>
            ))}
            <li
              className={`px-3 h-8 border border-[#c9c9c9] flex items-center justify-center bg-white text-black text-sm font-medium rounded cursor-pointer ${currentPage === totalPages ? 'opacity-50 cursor-not-allowed' : ''}`}
              onClick={() => handlePageChange(currentPage + 1)}
            >
              Next
            </li>
          </ul>
        </div>
      </div>
      {isOpen && (
        <Modal>
          <div
            className="relative max-w-xl w-full bg-white rounded py-7 px-10 shadow-[0_0_40px_rgba(0,0,0,0.25)]"
            ref={modalRef}
          >
            {/* Close Button */}
            <button
              type="button"
              onClick={() => setIsOpen(false)}
              className="absolute top-3 right-3 text-gray-400 hover:text-gray-700 text-2xl font-bold focus:outline-none"
              aria-label="Close"
            >
              &times;
            </button>
            <label className="mb-2 block text-sm font-medium text-gray-700">
              Write a review
            </label>
            <div className="relative leading-0 mb-5">
              <textarea rows={6} placeholder="" className="w-full bg-[#f8f6f6]/[.35] border border-[#c4c4c4] rounded p-4 text-base resize-none"></textarea>
            </div>
            <div className="flex justify-end">
              <button type="button" onClick={() => setIsOpen(false)} className="cursor-pointer bg-primary text-white text-[15px] font-normal h-12 rounded min-w-40">Submit</button>
            </div>
          </div>
        </Modal>
      )}
    </Layout>
  );
}