"use client";
import Layout from "@/components/Layout/Layout";
import axios from "axios";
import { useAuth } from "@/context/AuthContext";
import { useEffect, useState } from "react";
import { formatDateWithOrdinal } from "@/utils/dateFormat";
import PasswordChangeModal from "@/components/Settings/PasswordChangeModal";

export default function Settings() {
  const { user, loading } = useAuth();
  const [results, setResults] = useState({});
  const [fetching, setFetching] = useState(false);
  const [name, setName] = useState('');
  const [saving, setSaving] = useState(false);
  const [isModalOpen, setIsModalOpen] = useState(false);

  const apiUrl = process.env.NEXT_PUBLIC_API_URL;


  useEffect(() => {
    const fetchSettings = async () => {
      if (!user?.token || !user?.userId) return;
      setFetching(true);
      try {
        const response = await axios.get(`${apiUrl}/users/${user.userId}`, {
          headers: {
            Authorization: `Bearer ${user.token}`
          }
        });
        setResults(response.data || {});
        setName(response.data.name);
      } catch (err) {
        setResults({});
      }
      setFetching(false);
    };

    if (!loading && user?.token && user?.userId) {
      fetchSettings();
    }
  }, [user, loading, apiUrl]);

  const handleSubmit = async () => {
    if (!user?.token || !user?.userId) return;
    try {
      setSaving(true);
      const response = await axios.put(`${apiUrl}/users/${user.userId}`,
        { name: name },
        {
          headers: {
            Authorization: `Bearer ${user.token}`
          }
        }
      );
      if (response.status === 200) {
        alert("Settings updated successfully");
      } else {
        alert("Failed to update settings");
      }
      setSaving(false);
    }
    catch (err) {
      console.error("Error updating settings:", err);
      alert("Failed to update settings");
      setSaving(false);
    }
  };

  if (loading || fetching || !user) {
    return (
      <Layout>
        <div className="max-w-3xl mx-auto my-10">
          <h3 className="mb-5 font-semibold text-[#373737] text-2xl">Settings</h3>
          <div>Loading...</div>
        </div>
      </Layout>
    );
  }

  return (
    <Layout>
      <div className="max-w-3xl mx-auto my-10">
        <h3 className="mb-5 font-semibold text-[#373737] text-2xl">Settings</h3>
        <div className='border border-[#E8E8E8]'>
          <div className='bg-[#ECEFF6] h-[44px] w-full'></div>
          <div className='p-6'>
            <div className="mb-4">
              <label className="block text-sm font-medium text-gray-700">Name</label>
              <input
                type="text"
                value={name || ''}
                onChange={(e) => setName(e.target.value)}
                className="w-full border border-[#d1d1d1] rounded h-[52px] px-4"
              />
            </div>
            <div className="mb-4">
              <label className="block text-sm font-medium text-gray-700">Email</label>
              <input
                type="email"
                value={results.email || ''}
                readOnly
                className="w-full border border-[#d1d1d1] rounded h-[52px] px-4"
              />
            </div>
            <div className="mb-4">
              <label className="block text-sm font-medium text-gray-700">Plan Expiry Date </label>
              {results.expiryDate ? (
                <span>{formatDateWithOrdinal(results.expiryDate)}</span>
              ) : (
                <span className="text-red-500">Not subscribed</span>
              )}
            </div>
            <div className="flex items-center gap-4 mb-4">
              <button
                onClick={() => handleSubmit()}
                className="px-4 py-2 bg-primary text-white rounded hover:bg-primaryDark transition"
              >{saving? 'Saving...' : 'Save Changes'}</button>
              <button
                onClick={() => setIsModalOpen(true)}
                className="px-4 py-2 bg-gray-600 text-white rounded hover:bg-gray-700 transition"
              >
                Change Password
              </button>
              </div>
          </div>
        </div>
      </div>
      <PasswordChangeModal isOpen={isModalOpen} onClose={() => setIsModalOpen(false)} />
    </Layout>
  );
}