'use client';
import Image from "next/image";
import {   droparrow, filter } from '@/assets/images'
import {
  AreaChart,
  Area,
  XAxis,
  YAxis,
  CartesianGrid,
  Tooltip,
  ResponsiveContainer,
} from 'recharts';
import { useEffect, useState } from "react";
import axios from "axios";
import { useAuth } from "@/context/AuthContext";

const data = [
  { day: '01', value: 10000 },
  { day: '02', value: 70000 },
  { day: '03', value: 30000 },
  { day: '04', value: 50000 },
  { day: '05', value: 40000 },
  { day: '06', value: 90000 },
  { day: '07', value: 60000 },
  { day: '08', value: 70000 },
  { day: '09', value: 55000 },
  { day: '10', value: 60000 },
  { day: '11', value: 30000 },
  { day: '12', value: 1500000 },
  { day: '13', value: 80000 },
  { day: '14', value: 2000000 },
  { day: '15', value: 95000 },
  { day: '16', value: 70000 },
  { day: '17', value: 50000 },
  { day: '18', value: 1100000 },
  { day: '19', value: 85000 },
  { day: '20', value: 1000000 },
  { day: '21', value: 95000 },
  { day: '22', value: 80000 },
  { day: '23', value: 70000 },
  { day: '24', value: 60000 },
  { day: '25', value: 85000 },
  { day: '26', value: 30000 },
  { day: '27', value: 95000 },
  { day: '28', value: 85000 },
  { day: '29', value: 120000 },
  { day: '30', value: 2500000 },
  { day: '31', value: 70000 },
];

export default function SearchActivityChart() {
  const { user, loading: authLoading } = useAuth();
  const [chartData, setChartData] = useState([]);
  const [chartLoading, setChartLoading] = useState(true);
  const [chartError, setChartError] = useState(null);

  const apiUrl = process.env.NEXT_PUBLIC_API_URL;

  useEffect(() => {
    const fetchChartData = async () => {
      if (authLoading || !user?.token || !user?.userId) {
        setChartLoading(false);
        return;
      }

      setChartLoading(true);
      setChartError(null);
      try {
        const response = await axios.get(`${apiUrl}/getdata/search-by-days`, {
          headers: {
            Authorization: `Bearer ${user.token}`,
          },
          params: {
            authID: user.userId,
          },
        });
        setChartData(response.data);
      } catch (err) {
        console.error("Error fetching chart data:", err);
        setChartError("Failed to load chart data.");
      } finally {
        setChartLoading(false);
      }
    };

    fetchChartData();
  }, [user, authLoading, apiUrl]);

  // Calculate max value for YAxis
  const maxValue = chartData.length > 0
    ? Math.max(...chartData.map(d => d.value))
    : 10000; // fallback if no data

  if (chartLoading) {
    return (
      <div className="bg-white p-6 rounded-lg shadow border border-[#E8E8E8] text-center">
        Loading chart data...
      </div>
    );
  }

  if (chartError) {
    return (
      <div className="bg-white p-6 rounded-lg shadow border border-[#E8E8E8] text-center text-red-500">
        {chartError}
      </div>
    );
  }

  return (
    <div className="bg-white p-6 rounded-lg shadow border border-[#E8E8E8]">
      <div className="flex justify-between items-center mb-4">
        <h2 className="text-lg font-medium text-gray-800">
          Search Activity (Last 30 Days)
        </h2>
        <div className="flex gap-2">
          <button className="px-3 flex items-center py-1.5 border border-gray-300 rounded text-sm text-gray-700 hover:bg-gray-50 transition">
            Filter <Image src={filter} className="ml-2" alt=""/>
          </button>
          <button className="px-3 flex items-center py-1.5 border border-gray-300 rounded text-sm text-gray-700 hover:bg-gray-50 transition">
            This Month  <Image src={droparrow} className="ml-2" alt=""/>
          </button>
        </div>
      </div>

      <div className="w-full h-64"> {/* Add pl-4 or pl-6 for spacing */}
        <ResponsiveContainer width="100%" height="100%">
          <AreaChart
            data={chartData}
            className="text-[12px]"
            margin={{ left: 0, right: 0, top: 10, bottom: 0 }}
          >
            <defs>
              <linearGradient id="colorValue" x1="0" y1="0" x2="0" y2="1">
                <stop offset="5%" stopColor="#10b981" stopOpacity={0.3} />
                <stop offset="95%" stopColor="#10b981" stopOpacity={0} />
              </linearGradient>
            </defs>

            <XAxis dataKey="date" tickLine={false} />
            <YAxis
                width={50}
                domain={[0, maxValue]}
                tickFormatter={(value) => {
                    if (value >= 1000000) return `${value / 1000000}M`;
                    if (value >= 1000) return `${value / 1000}K`;
                    return value;
                }}
                tickLine={false}
                axisLine={false}
            />

            <CartesianGrid strokeDasharray="3 3" vertical={false} />
            <Tooltip
              formatter={(value) =>
                typeof value === 'number' ? value.toLocaleString() : value
              }
            />
            <Area
              type="monotone"
              dataKey="value"
              stroke="#10b981"
              fillOpacity={1}
              fill="url(#colorValue)"
              strokeWidth={2}
            />
          </AreaChart>
        </ResponsiveContainer>
      </div>
    </div>
  );
}
